<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void {

        $prefix = 'company_';

        // ---------------------
        // Companies (Tenants)
        // ---------------------
        Schema::create('companies', function (Blueprint $table) {
            $table->id();

            // Basic Info
            $table->string('name');
            $table->string('legal_name')->nullable();
            $table->string('registration_no')->nullable();
            $table->string('tax_id')->nullable();

            // Contact
            $table->string('email')->nullable();
            $table->string('phone')->nullable();
            $table->string('website')->nullable();

            // Address
            $table->string('country')->nullable();
            $table->string('state')->nullable();
            $table->string('city')->nullable();
            $table->string('address')->nullable();
            $table->string('postal_code')->nullable();

            // Branding
            $table->string('logo')->nullable();
            $table->string('theme')->default('default');

            // System Defaults
            $table->string('timezone', 100)->default('UTC');
            $table->string('date_format', 50)->default('Y-m-d');

            $table->boolean('is_active')->default(true);
            $table->timestampsTz();
            $table->softDeletesTz();
        });

        // ---------------------
        // Branches
        // ---------------------
        Schema::create($prefix.'branches', function (Blueprint $table) {
            $table->id();

            // Remove FK – store company_id as normal bigint
            $table->unsignedBigInteger('company_id')->nullable();

            $table->string('name');
            $table->string('code')->nullable();
            $table->string('type')->default('store');

            // Address
            $table->string('country')->nullable();
            $table->string('state')->nullable();
            $table->string('city')->nullable();
            $table->string('address')->nullable();

            // Geo support
            $table->decimal('latitude', 10, 6)->nullable();
            $table->decimal('longitude', 10, 6)->nullable();

            $table->boolean('is_active')->default(true);
            $table->boolean('is_sync_allowed')->default(true);

            $table->timestampsTz();
            $table->softDeletesTz();
        });

        // ---------------------
        // Plans
        // ---------------------
        Schema::create($prefix.'plans', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->decimal('price_monthly', 15, 2)->nullable();
            $table->decimal('price_yearly', 15, 2)->nullable();
            $table->text('description')->nullable();
            $table->json('features')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestampsTz();
        });

        // ---------------------
        // Subscriptions
        // ---------------------
        Schema::create($prefix.'subscriptions', function (Blueprint $table) use ($prefix) {
            $table->id();

            $table->unsignedBigInteger('company_id')->nullable();
            $table->unsignedBigInteger('plan_id')->nullable();

            $table->enum('billing_cycle', ['monthly', 'yearly'])->default('monthly');
            $table->date('starts_at')->nullable();
            $table->date('ends_at')->nullable();

            $table->boolean('is_active')->default(true);
            $table->boolean('auto_renew')->default(true);
            $table->timestampsTz();
        });

        // ---------------------
        // Company Limits
        // ---------------------
        Schema::create($prefix.'limits', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('company_id')->nullable();

            $table->integer('max_users')->nullable();
            $table->integer('max_branches')->nullable();
            $table->integer('max_products')->nullable();
            $table->integer('max_invoices_per_month')->nullable();
            $table->integer('max_customers')->nullable();
            $table->integer('max_suppliers')->nullable();
            $table->integer('max_storage_mb')->nullable();

            $table->timestampsTz();
        });

        // ---------------------
        // Company Modules
        // ---------------------
        Schema::create($prefix.'modules', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('company_id')->nullable();

            $table->string('module_key');
            $table->boolean('is_enabled')->default(true);

            $table->timestampsTz();

            // the unique can remain because it's not an FK relation
            $table->unique(['company_id', 'module_key']);
        });

        // ---------------------
        // Company Billing History
        // ---------------------
        Schema::create($prefix.'billing_history', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('company_id')->nullable();
            $table->unsignedBigInteger('plan_id')->nullable();

            $table->string('invoice_no')->nullable();
            $table->decimal('amount', 15, 2)->default(0);
            $table->string('currency', 10)->default('USD');

            $table->enum('payment_method', ['stripe','paypal','razorpay','manual'])->nullable();
            $table->enum('status', ['paid','unpaid','failed'])->default('unpaid');

            $table->timestampsTz();
        });

        // ---------------------
        // Billing Items
        // ---------------------
        Schema::create($prefix.'billing_history_items', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('billing_id')->nullable();

            $table->string('item_name');
            $table->decimal('amount', 15, 2);
            $table->string('currency', 10)->default('USD');
            $table->integer('quantity')->default(1);

            $table->timestampsTz();
        });

        // ---------------------
        // Custom Services / Add-ons
        // ---------------------
        Schema::create($prefix.'custom_services', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('company_id')->nullable();

            $table->string('service_name');
            $table->decimal('price', 15, 2);
            $table->enum('billing_cycle', ['monthly', 'yearly'])->default('monthly');
            $table->boolean('is_active')->default(true);

            $table->timestampsTz();
        });

        // ---------------------
        // Audit Logs
        // ---------------------
        Schema::create($prefix.'audit_logs', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('company_id')->nullable();

            // No FK
            $table->unsignedBigInteger('user_id')->nullable();

            $table->string('module');
            $table->string('action');
            $table->json('old_data')->nullable();
            $table->json('new_data')->nullable();
            $table->timestampsTz();
        });

        // ---------------------
        // Company Tax Settings
        // ---------------------
        Schema::create($prefix.'tax_settings', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('company_id')->nullable();
            $table->string('tax_name')->nullable();
            $table->decimal('tax_rate', 5, 2)->default(0);
            $table->string('tax_type')->default('vat');
            $table->timestampsTz();
        });
    }

    public function down(): void {
        $prefix = 'company_';

        Schema::dropIfExists($prefix.'tax_settings');
        Schema::dropIfExists($prefix.'audit_logs');
        Schema::dropIfExists($prefix.'custom_services');
        Schema::dropIfExists($prefix.'billing_history_items');
        Schema::dropIfExists($prefix.'billing_history');
        Schema::dropIfExists($prefix.'modules');
        Schema::dropIfExists($prefix.'limits');
        Schema::dropIfExists($prefix.'subscriptions');
        Schema::dropIfExists($prefix.'plans');
        Schema::dropIfExists($prefix.'branches');
        Schema::dropIfExists('companies');
    }
};
