<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {

    protected string $p = 'install_';

    public function up(): void
    {
        $p = $this->p;

        /*-------------------------------------------------------
        | 1️⃣ Loan Products
        -------------------------------------------------------*/
        Schema::create($p.'products', function (Blueprint $table) {
            $table->id();
            $table->string('name'); 
            $table->decimal('interest_rate', 5, 2)->default(0);
            $table->enum('interest_type', ['flat', 'reducing'])->default('flat');
            $table->integer('max_duration_months')->default(12);
            $table->decimal('min_amount', 12, 2)->default(0);
            $table->decimal('max_amount', 12, 2)->default(0);
            $table->boolean('is_active')->default(true);
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 2️⃣ Loan Applications
        -------------------------------------------------------*/
        Schema::create($p.'applications', function (Blueprint $table) {
            $table->id();

            $table->unsignedBigInteger('customer_id'); // removed FK
            $table->unsignedBigInteger('product_id');  // removed FK

            $table->decimal('requested_amount', 12, 2);
            $table->integer('requested_months');

            $table->enum('status', ['pending', 'approved', 'rejected', 'cancelled'])
                ->default('pending');

            $table->text('remarks')->nullable();
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 3️⃣ Loans
        -------------------------------------------------------*/
        Schema::create($p.'loans', function (Blueprint $table) use ($p) {
            $table->id();
            $table->uuid('uuid')->unique();

            $table->unsignedBigInteger('application_id')->nullable(); // removed FK
            $table->unsignedBigInteger('customer_id'); // removed FK
            $table->unsignedBigInteger('product_id');  // removed FK

            $table->decimal('loan_amount', 12, 2);
            $table->decimal('down_payment', 12, 2)->default(0);
            $table->decimal('net_financed', 12, 2); 

            $table->decimal('interest_rate', 5, 2)->default(0); 
            $table->enum('interest_type', ['flat', 'reducing'])->default('flat');

            $table->integer('duration_months');
            $table->decimal('monthly_installment', 12, 2); 

            $table->date('start_date');

            $table->enum('status', ['active', 'completed', 'defaulted', 'written_off'])
                  ->default('active');

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 4️⃣ Loan Guarantors
        -------------------------------------------------------*/
        Schema::create($p.'guarantors', function (Blueprint $table) use ($p) {
            $table->id();

            $table->unsignedBigInteger('loan_id'); // removed FK
            $table->unsignedBigInteger('guarantor_id'); // removed FK

            $table->string('relation')->nullable();
            $table->string('phone')->nullable();

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 5️⃣ Loan Collateral
        -------------------------------------------------------*/
        Schema::create($p.'collateral', function (Blueprint $table) use ($p) {
            $table->id();
            $table->unsignedBigInteger('loan_id'); // removed FK

            $table->string('type');
            $table->string('description')->nullable();
            $table->decimal('estimated_value', 12, 2)->nullable();

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 6️⃣ Installment Schedule
        -------------------------------------------------------*/
        Schema::create($p.'schedules', function (Blueprint $table) use ($p) {
            $table->id();
            $table->unsignedBigInteger('loan_id'); // removed FK

            $table->unsignedInteger('installment_no');
            $table->date('due_date');

            $table->decimal('amount', 12, 2);
            $table->decimal('principal', 12, 2)->nullable(); 
            $table->decimal('interest', 12, 2)->nullable();

            $table->enum('status', ['pending', 'paid', 'late', 'partial'])
                  ->default('pending');

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 7️⃣ Payments
        -------------------------------------------------------*/
        Schema::create($p.'payments', function (Blueprint $table) use ($p) {
            $table->id();
            $table->uuid('uuid')->unique();

            $table->unsignedBigInteger('loan_id'); // removed FK
            $table->unsignedBigInteger('schedule_id')->nullable(); // removed FK
            $table->unsignedBigInteger('collector_id')->nullable(); // removed FK

            $table->decimal('amount', 12, 2);
            $table->decimal('penalty', 12, 2)->default(0);

            $table->enum('method', ['cash','bank','online','mobile_wallet']);
            $table->string('reference')->nullable();

            $table->date('payment_date');
            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 8️⃣ Penalty / Waivers
        -------------------------------------------------------*/
        Schema::create($p.'penalties', function (Blueprint $table) use ($p) {
            $table->id();

            $table->unsignedBigInteger('schedule_id'); // removed FK
            $table->decimal('amount', 12, 2);
            $table->boolean('is_waived')->default(false);
            $table->text('remarks')->nullable();

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 9️⃣ Loan Reschedule Requests
        -------------------------------------------------------*/
        Schema::create($p.'reschedules', function (Blueprint $table) use ($p) {
            $table->id();
            $table->unsignedBigInteger('loan_id'); // removed FK

            $table->enum('status', ['pending','approved','rejected'])->default('pending');
            $table->integer('extended_months')->default(0);
            $table->decimal('new_installment', 12, 2)->nullable();
            $table->text('reason')->nullable();

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 🔟 Recovery Visits
        -------------------------------------------------------*/
        Schema::create($p.'recovery_visits', function (Blueprint $table) use ($p) {
            $table->id();
            $table->unsignedBigInteger('loan_id'); // removed FK
            $table->unsignedBigInteger('collector_id')->nullable(); // removed FK

            $table->date('visit_date');
            $table->text('notes')->nullable();

            $table->timestampsTz();
        });

        /*-------------------------------------------------------
        | 1️⃣1️⃣ Loan Ledger
        -------------------------------------------------------*/
        Schema::create($p.'ledger', function (Blueprint $table) use ($p) {
            $table->id();

            $table->unsignedBigInteger('loan_id'); // removed FK
            $table->enum('type', ['debit', 'credit']);
            $table->string('reason');
            $table->decimal('amount', 12, 2);
            $table->date('entry_date');

            $table->timestampsTz();
        });
    }

    public function down(): void
    {
        $p = $this->p;

        Schema::dropIfExists($p.'ledger');
        Schema::dropIfExists($p.'recovery_visits');
        Schema::dropIfExists($p.'reschedules');
        Schema::dropIfExists($p.'penalties');
        Schema::dropIfExists($p.'payments');
        Schema::dropIfExists($p.'schedules');
        Schema::dropIfExists($p.'collateral');
        Schema::dropIfExists($p.'guarantors');
        Schema::dropIfExists($p.'loans');
        Schema::dropIfExists($p.'applications');
        Schema::dropIfExists($p.'products');
    }
};
